﻿using Hl7.Fhir.Model;
using log4net;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Threading.Tasks;
using VA.PPMS.IWS.Api.Configuration.Interface;
using VA.PPMS.IWS.Api.Data;
using VA.PPMS.IWS.Api.Helpers;
using VA.PPMS.IWS.Api.HttpClientHandler.Interface;
using VA.PPMS.IWS.Common;

namespace VA.PPMS.IWS.Api.Controllers
{
    [Route("OperationOutcome/v1")]
    public class DasOperationOutcomeController : Controller
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IHttpClientHandler _client;

        public DasOperationOutcomeController(ILog logger, IIwsConfiguration configuration, IHttpClientHandler client)
        {
            _logger = logger;
            _configuration = configuration;
            _client = client;
        }

        [HttpPost]
        public async Task<IActionResult> Post([FromBody]string fhirMessage)
        {
            string conversationId = null;

            try
            {
                _logger.Info("@@@@@ Info - Start POST for DasOperationOutcomeController @@@@@");
                _logger.Info($"---- DEBUG: Message = {fhirMessage} ");

                var message = DasHelper.ParseHeaders(Request);
                var outcome = FhirHelper.ConvertToOutcome(fhirMessage);
                conversationId = message.Id;

                if (outcome == null)
                {
                    _logger.Error("DasOperationOutcomeController: OperationOutcome payload is missing");
                    return StatusCode(StatusCodes.Status400BadRequest);
                }

                var hasOutcomeId = !string.IsNullOrWhiteSpace(outcome.Id);
                if (hasOutcomeId && message.IsValid)
                {
                    _logger.Info($"-- DasOperationOutcome - start process data: {outcome.Id}");

                    message.Content = OutcomeToString(outcome);
                    _logger.Info($"-- DasOperationOutcome - Content: {message.Content}");

                    using (var response = await _client.PostAsync(await _configuration.GetDasOperationOutcomeUri(), message))
                    {
                        if (response.IsSuccessStatusCode)
                        {
                            _logger.Info("@@@@@ Info - End POST for DasOperationOutcomeController @@@@@");
                            return StatusCode(StatusCodes.Status201Created, DasHelper.StatusCode201(conversationId));
                        }

                        _logger.Error($"DasOperationOutcomeController POST Exception. There was an error posting to the Function API: {response}");
                        var error = await response.Content.ReadAsStringAsync();
                        _logger.Error($"DasOperationOutcomeController POST Exception. The error is {error}: {response}");

                        return StatusCode(StatusCodes.Status400BadRequest, DasHelper.StatusCode400(conversationId, message.ReceiverId));
                    }
                }

                _logger.Error("CcnInitialOutcomeController: DasOperationOutcomeController is missing data required to process the information");
                return StatusCode(StatusCodes.Status400BadRequest, DasHelper.StatusCode400(conversationId, message.ReceiverId));
            }
            catch (Exception ex)
            {
                _logger.Error("DasOperationOutcomeController POST Exception:", ex);
                return StatusCode(StatusCodes.Status500InternalServerError, DasHelper.StatusCode500(conversationId));
            }
        }

        private string OutcomeToString(OperationOutcome outcome)
        {
            string message = string.Empty;

            if (outcome != null)
            {
                var das = new DasOutcome()
                {
                    Id = outcome.Id,
                };

                if (outcome.Issue != null && outcome.Issue.Count > 0)
                {
                    var issue = outcome.Issue[0];

                    das.Severity = issue.Severity?.ToString();
                    das.Code = issue.Code?.ToString();
                    das.Details = issue.Details?.Text;
                }

                message = das.ToString();
            }

            return message;
        }
    }
}